//------------------------
// PARAMETERS

float4x4 gWorldViewProjMatrix
<
   string paramClass="intrinsic";
   string paramName="worldViewProj";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float4x4 gWorldViewMatrix
<
   string paramClass="intrinsic";
   string paramName="worldView";
> = float4x4(1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1);

float gRGBScale
<
   string paramClass="intrinsic";
   string paramName="RGBScale";
   string paramType="float";
> = 1.0f;

float gFogDensity
<
   string paramClass="intrinsic";
   string paramName="FogDensity";
   string paramType="float";
> = 0.0f;

float gFogStart
<
   string paramClass="intrinsic";
   string paramName="FogStart";
   string paramType="float";
> = 0.0;

float gPlanarFogDensity
<
  string paramClass="intrinsic";
  string paramName="PlanarFogDensity";
  string paramType="float";
> = 0.0;

float4 gPlanarFogEquation
<
  string paramClass="intrinsic";
  string paramName="PlanarFogEquation";
> = float4(0,1,0,0);

float4 gPlanarFogColor
<
string paramClass="intrinsic";
string paramName="PlanarFogColor";
string paramType="color";
> = float4(0,0,0,1);

float4 gFogColor
<
   string paramClass="intrinsic";
   string paramName="FogColor";
   string paramType="color";
> = float4(0,0,0,1);

sampler tex0   : register(s0);
sampler tex1   : register(s1);
sampler tex2   : register(s2);
sampler tex3   : register(s3);

// END PARAMETERS


//------------------------
// SUBROUTINES

vec4 SRGBToLinear(vec4 c)
{
   vec4 ret = c;
   ret.rgb *= ret.rgb;
   return ret;
}

float ComputeSquaredExponentialFogDensity(float fogDistance)
{
   float  squaredExponent = fogDistance * fogDistance;      
   float  finalFogDensity = clamp( exp(-squaredExponent), 0.0, 1.0);
   return finalFogDensity;   
}

// END SUBROUTINES


//---------------------------
// FUNCTION_VS WaveFoam30VS

uniform mat4  gWorldViewProjMatrix;
uniform mat4  gWorldViewMatrix;
uniform float gFogDensity;
uniform float gFogStart;
uniform float gPlanarFogDensity;
uniform vec4  gPlanarFogEquation;

varying vec2  FogParams;

void main()
{
	// Transform position
	gl_Position = gl_Vertex * gWorldViewProjMatrix;

	// Diffuse vertex color
	gl_FrontColor = gl_Color;

	gl_TexCoord[0].st = gl_MultiTexCoord0.st;
	gl_TexCoord[1].st = gl_MultiTexCoord1.st;
	gl_TexCoord[2].st = gl_MultiTexCoord2.st;
	gl_TexCoord[3].st = gl_MultiTexCoord3.st;

	// Compute Fog

	vec4 viewSpacePos = gl_Vertex * gWorldViewMatrix;		// view space (no projection)
	
	FogParams.x = ComputeSquaredExponentialFogDensity( gFogDensity * max(0.0, viewSpacePos.z-gFogStart) / 1000.0);

	float perpDist = dot(viewSpacePos.xyz, gPlanarFogEquation.xyz);
	perpDist -= gPlanarFogEquation.w;

	float t;
	if( perpDist > 0.0 )
		t = 0.0;
	else
	{
		vec3  rayDirection = normalize(-viewSpacePos.xyz);
		float dirAlong     = dot(gPlanarFogEquation.xyz, rayDirection);
		float posAlong     = dot(gPlanarFogEquation.xyz, viewSpacePos.xyz);

		if (abs(dirAlong) < 0.00125)
			dirAlong = 1000.0;
		else
			dirAlong = -1.0 / dirAlong;

		t = gPlanarFogEquation.w - posAlong;
		t *= dirAlong;
	}

	FogParams.y = ComputeSquaredExponentialFogDensity( gPlanarFogDensity * t * 0.01 );   
}


//--------------------------
// FUNCTION_PS WaveFoam30PS

uniform sampler2D tex0;
uniform sampler2D tex1;
uniform sampler2D tex2;
uniform sampler2D tex3;

#ifdef SCALERGBOUTPUT
uniform float gRGBScale;
#endif
uniform vec4 gPlanarFogColor;
uniform vec4 gFogColor;

varying vec2 FogParams;

void main()
{
	vec4 color = SRGBToLinear( texture2D(tex0, gl_TexCoord[0].st) );
	color *= SRGBToLinear( texture2D(tex1, gl_TexCoord[1].st) );
	color *= SRGBToLinear( texture2D(tex2, gl_TexCoord[2].st) );
	color *= SRGBToLinear( texture2D(tex3, gl_TexCoord[3].st) );

	color *= gl_Color;

	color.rgb = mix( gPlanarFogColor.rgb, color.rgb, FogParams.y );
	color.rgb = mix( gFogColor.rgb, color.rgb, FogParams.x );

#ifdef SCALERGBOUTPUT
	color.rgb *= gRGBScale;
#endif   

	gl_FragColor = color;
}

